<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Producto;
use App\Models\Venta;
use App\Models\MovimientoStock;

class ReportController extends Controller
{
    public function pdf(Request $request, $type)
    {
        $title = "REPORTE DE " . strtoupper($type);
        $columns = [];
        $summary = [];

        switch ($type) {
            case 'productos':
                $query = Producto::with('categoria')->get();
                $columns = ['Código', 'Producto', 'Categoría', 'Stock', 'Mín.', 'Precio'];
                $data = $query->map(fn($p) => [$p->codigo, $p->nombre, $p->categoria?->nombre, $p->stock, $p->stock_minimo, 'S/.' . number_format($p->precio, 2)]);
                $summary = [
                    'Total Productos' => $query->count(),
                    'Stock Crítico' => $query->whereColumn('stock', '<=', 'stock_minimo')->count(),
                    'Valor Total' => 'S/. ' . number_format($query->sum(fn($p) => $p->stock * $p->precio), 2)
                ];
                break;
            case 'ventas':
                $query = Venta::with('usuario')->orderBy('created_at', 'desc')->get();
                $columns = ['Ticket', 'Fecha', 'Cajero', 'Doc.', 'Total'];
                $data = $query->map(fn($v) => ['#' . $v->id, $v->created_at->format('d/m/Y H:i'), $v->usuario?->name, ucfirst($v->tipo_documento), 'S/.' . number_format($v->total, 2)]);
                $summary = [
                    'Cantidad Ventas' => $query->count(),
                    'Total Recaudado' => 'S/. ' . number_format($query->sum('total'), 2)
                ];
                break;
            case 'kardex':
                $query = MovimientoStock::with(['producto', 'usuario'])->orderBy('created_at', 'desc')->get();
                $columns = ['Fecha', 'Producto', 'Tipo', 'Cant.', 'Motivo', 'Usuario'];
                $data = $query->map(fn($m) => [$m->created_at->format('d/m/Y H:i'), $m->producto?->nombre, strtoupper($m->tipo), $m->cantidad, $m->motivo, $m->usuario?->name]);
                $summary = [
                    'Total Movimientos' => $query->count(),
                    'Entradas' => $query->where('tipo', 'entrada')->sum('cantidad'),
                    'Salidas' => $query->where('tipo', 'salida')->sum('cantidad')
                ];
                break;
        }

        return view('reports.print', compact('title', 'columns', 'data', 'summary'));
    }

    public function ticket(Venta $venta)
    {
        $venta->load(['detalles.producto', 'usuario']);
        return view('reports.ticket', compact('venta'));
    }

    public function excel($type)
    {
        $title = "REPORTE DE " . strtoupper($type);
        $filename = "reporte_{$type}_" . date('Ymd_His') . ".xls";
        
        $columns = [];
        $data = [];
        $stats = [];

        switch ($type) {
            case 'productos':
                $query = Producto::with('categoria')->get();
                $columns = ['CODIGO', 'PRODUCTO', 'CATEGORIA', 'STOCK', 'PRECIO'];
                $data = $query->map(fn($p) => [$p->codigo, $p->nombre, $p->categoria?->nombre, $p->stock, $p->precio]);
                $stats = ["Total Productos: " . $query->count(), "Stock Crítico: " . $query->whereColumn('stock', '<=', 'stock_minimo')->count()];
                break;
            case 'ventas':
                $query = Venta::with('usuario')->get();
                $columns = ['TICKET', 'FECHA', 'CAJERO', 'DOCUMENTO', 'TOTAL'];
                $data = $query->map(fn($v) => ['#' . $v->id, $v->created_at->format('d/m/Y H:i'), $v->usuario?->name, strtoupper($v->tipo_documento), (float)$v->total]);
                $stats = ["Ventas Realizadas: " . $query->count(), "Total: S/. " . number_format($query->sum('total'), 2)];
                break;
            case 'kardex':
                $query = MovimientoStock::with('producto')->get();
                $columns = ['FECHA', 'PRODUCTO', 'TIPO', 'CANTIDAD', 'MOTIVO'];
                $data = $query->map(fn($m) => [$m->created_at->format('d/m/Y H:i'), $m->producto?->nombre, strtoupper($m->tipo), $m->cantidad, $m->motivo]);
                $stats = ["Entradas: " . $query->where('tipo', 'entrada')->sum('cantidad'), "Salidas: " . $query->where('tipo', 'salida')->sum('cantidad')];
                break;
        }

        $html = '<html><head><meta charset="UTF-8"></head><body>';
        $html .= '<table border="1">';
        $html .= '<tr><th colspan="'.count($columns).'" style="background-color:#0f172a; color:#ffffff; font-size:16pt; height:40px;">CHELYFARMA</th></tr>';
        $html .= '<tr><th colspan="'.count($columns).'" style="background-color:#1e293b; color:#ffffff; font-size:12pt;">' . $title . '</th></tr>';
        $html .= '<tr><th colspan="'.count($columns).'" style="text-align:left; color:#64748b;">Generado: ' . now()->format('d/m/Y H:i') . '</th></tr>';
        
        $html .= '<tr>';
        foreach ($columns as $col) $html .= '<th style="background-color:#f1f5f9; font-weight:bold;">' . $col . '</th>';
        $html .= '</tr>';
        
        foreach ($data as $row) {
            $html .= '<tr>';
            foreach ($row as $cell) $html .= '<td>' . $cell . '</td>';
            $html .= '</tr>';
        }

        $html .= '<tr><td colspan="'.count($columns).'"></td></tr>';
        foreach ($stats as $stat) {
            $html .= '<tr><th colspan="'.count($columns).'" style="text-align:right; background-color:#f8fafc;">' . $stat . '</th></tr>';
        }
        $html .= '</table></body></html>';

        return response($html)->header('Content-Type', 'application/vnd.ms-excel')->header('Content-Disposition', 'attachment; filename="'.$filename.'"');
    }
}